CODES	SEGMENT
	ASSUME	CS:CODES,DS:CODES,ES:CODES
;============================================================================
;
;File:	Z80EM1.ASM - Z80 Emulator
;Date:  April 20, 1987
;
;============================================================================
;
;						      Location
;   Register Usage:	AX = Z80 A-PSW	(AH=F, AL=A)	FFD6H
;			BX = Z80 H-L	(BH=H, BL=L)	FFD0H
;			DI = Z80 B-C			FFD4H
;			DX = Z80 D-E	(DH=D, DL=E)	FFD2H
;			BP = Z80 SP			
;			SI = Z80 P	(Program Cntr)
;		      	     Z80 IX			FFD8H
;			     Z80 IY			FFDAH
;			     Z80 I			FFDEH
;			     Z80 R			FFDFH
;
;	Z80 Alternate Register Set 
;
	ORG	0FFD0H
HL	DW	0
DE	DW 	0
BC	DW 	0
AF	EQU	$	;A is 8-bit accumulator
PSW	DW	0
;
; Z80 Primary Registers (emulated in memory)
IX	DW 	0
IY	DW 	0
IZ	DW	0	;save of (IX+disp) or (IY+disp)
I	DB	0	;Interrupt register
R	DB	0	;Refresh register
PC	DW	0	;PC saved
;
	ORG	0FFF4H
IS	DB	0	;Interrupt status byte
IM	DB	0	;Interrupt mode byte
;
INTFLG	DB	03H		;Z80 interrupt control (00H = disabled).
;			
;	ORG	0FF89H	documentation only	
;SLRFLG	DB	010H		;So SLR will run.
;
;
;---------------------------------------------------------------------------
;
PG	EQU	256		;Page boundary.
CBOP	EQU	002AH		;CB operations offset address        
DDOP	EQU	0054H		;DD operations offset address        
CBDDOP	EQU	00D2H		;DDCB / FDCB ops offset address      
EDOP	EQU	00A8H		;ED operations offset address         
FDOP	EQU	007EH		;FD operations offset address        
;
;---------------------------------------------------------------------------
;
;	NXTI - Next Instruction Macro Definition
;
NXTI	MACRO
	MOV	CH,[SI]		;get next instruction
	INC	SI		;increment program counter
	JMP	CX		;process next instruction
	ENDM
;
;--------------------------------------------------------------------------
;
PROG	SEGMENT
	ASSUME	CS:PROG
;
;--------------------------------------------------------------------------
;
	ORG	00H*PG		
NOP	PROC	NEAR		;00H - NOP		NOP
;				-----------------------------------
	XOR	CL,CL
	NXTI
NOP	ENDP
;
	ORG	01H*PG	
LXIB	PROC	NEAR		;01H - LXI B,D16	LD BC,D16
;				------------------------------------
	XCHG	DI,AX		;Save AX in DI		
	LODSW			;Get word pointed to by SI in AX, bump SI 
	XCHG	DI,AX		;AX to DI(B-C), and restore AX
	NXTI
LXIB	ENDP
;
;
	ORG	02H*PG
STAXB	PROC	NEAR		;02H - STAX B		LD (BC),A
;				-------------------------------------
	MOV	[DI],AL		;Store A(AL) at location DI (B-C)
	NXTI
STAXB	ENDP
;
	ORG	03H*PG
INXB	PROC	NEAR		;03H - INX B		INC BC
;				-------------------------------------
	INC	DI		;increment DI(B-C)
	NXTI
INXB	ENDP
;
	ORG	04H*PG
INRB	PROC	NEAR		;04H - INR B		INC B
;				-------------------------------------
	SAHF			;push flags from AH
	XCHG	DI,CX		;get B-C(DI) in CX (CH=B)
	INC	CH		;incement B (CH)
	LAHF			;pop flags to AH
	XCHG	DI,CX		;put B-C back in DI
	JO	INRB1		;jump on overflow
	AND	AH,0F9H		;clear parity flag
	NXTI
INRB1	EQU	$
	AND	AH,0F9H		
	OR	AH,004H		;set parity flag
	NXTI
INRB	ENDP
;
	ORG	05H*PG
DCRB	PROC	NEAR		;05H - DCR B		DEC B
;				--------------------------------------
	SAHF			;push flags from AH
	XCHG	DI,CX		;get B-C(DI) in CX (CH=B)
	DEC	CH		;decrement B (CH)
	LAHF			;pop flags to AH
	XCHG	DI,CX		;put B-C back in DI
	JO	DCRB1		;jump on overflow
	AND	AH,0F9H		;clear parity flag
	OR	AH,002H		;set subtract flag
	NXTI
DCRB1	EQU	$
	AND	AH,0F9H
	OR	AH,006H		;set parity flag
	NXTI
DCRB	ENDP	
;
	ORG	06H*PG
MVIB	PROC	NEAR		;06H - MVI B,D8		LD B,D8
;				----------------------------------------
	XCHG	DI,AX		;put B-C(DI) in AX (AH=B)
	MOV	AH,[SI]		;load B(AH) from immediate (SI) location.
	XCHG	DI,AX		;get result in B-C(DI)
	INC	SI		;increment program counter (SI)
	NXTI
MVIB	ENDP
;
	ORG	07H*PG
RLC	PROC	NEAR		;07H - RLC		RLCA
;				----------------------------------------
	SAHF			;push flags from AH
	ROL	AL,1		;rotate AL (A) left one bit
	LAHF			;load flags to AH
	AND	AH,0EDH		;clear aux carry and parity flags
	NXTI
RLC	ENDP
;
	ORG	08H*PG         
EXAF	PROC	NEAR		;08H - (none)		EX AF,AF'
;				-----------------------------------------
	XCHG	AX,[PSW]	;exchange AX(AF) with memory (AF')
	NXTI
EXAF	ENDP
;
	ORG	09H*PG
DADB	PROC	NEAR		;09H - DAD B		ADD HL,BC
;				-----------------------------------------
	AND	AH,0FCH		;clear carry flag
	ADD	BX,DI		;add HL(DI) to BC(BX)
	JB	DADB1		;jump on carry
	NXTI			;otherwise, get next instruction
DADB1	EQU	$
	INC	AH		;set carry flag
	NXTI
DADB	ENDP
;
	ORG	0AH*PG
LDAXB	PROC	NEAR		;0AH - LDAX B		LD A,(BC)
;				-----------------------------------------
	MOV	AL,[DI]		;Load A(AL) from address in BC(DI)
	NXTI
LDAXB	ENDP
;
	ORG	0BH*PG
DCXB	PROC	NEAR		;0BH - DCX B		DEC BC
;				-----------------------------------------
	DEC	DI		;Decrement BC(DI)
	NXTI
DCXB	ENDP
;
	ORG	0CH*PG
INRC	PROC	NEAR		;0CH - INR C		INC C
;				----------------------------------------
	SAHF			;store AH in flags
	XCHG	DI,CX		;save contents of BC(DI) in CX
	INC	CL		;increment C(CL)
	LAHF			;load AH from flags
	XCHG	DI,CX		;restore contents of BC(DI) from CX
	JO	INRC1		;jump on overflow
	AND	AH,0F9H		;clear parity/overflow flag
	NXTI
INRC1	EQU	$
	AND	AH,0F9H		;clear parity/overflow flag
	OR	AH,004H		;set parity/overflow flag
	NXTI
INRC	ENDP
;
	ORG	0DH*PG
DCRC	PROC	NEAR		;0DH - DCR C		DEC C
;				----------------------------------------
	SAHF			;store AH in flags
	XCHG	DI,CX		;save contents of BC(DI) in CX
	DEC	CL		;decrement C(CL)
	LAHF			;load AH from flags
	XCHG	DI,CX		;restore contents of BC(DI) from CX
	JO	DCRC1		;jump on overflow
	AND	AH,0F9H		;clear parity/overflow flag
	OR	AH,002H		;set Z80 subtract flag
	NXTI
DCRC1	EQU	$
	AND	AH,0F9H		;clear parity/overflow flag
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
DCRC	ENDP
;
	ORG	0EH*PG
MVIC	PROC	NEAR		;0EH - MVI C,D8		LD C,D8
;				-----------------------------------------
	XCHG	DI,AX		;exchange BC(DI) and PSW(AX)
	LODSB			;load A(AL) from immediate byte
	XCHG	DI,AX		;exchange BC(DI) and PSW(AX)
	NXTI
MVIC	ENDP
;
	ORG	0FH*PG
RRC	PROC	NEAR		;0FH - RRC		RRCA
;				-----------------------------------------
	SAHF			;store AH in flags
	ROR	AL,1		;rotate A(AL) right one bit
	LAHF			;load AH from flags
	AND	AH,0EDH		;clear aux carry and subtract flags
	NXTI
RRC	ENDP
;
	ORG	10H*PG
DJNZ	PROC	NEAR		;10H - (none)		DJNZ dd
;				-----------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX
	DEC	AH		;decrement B
	XCHG	DI,AX		;put result back in BC(DI)
	JZ	DJNZ1		;result is zero - go to next instruction
	XCHG	CX,AX		;result non-zero, so jump to dd (disp)
	LODSB			;get displacement byte
	CBW			;convert byte to word
	ADD	SI,AX		;add displacement to PC(SI)
	XCHG	CX,AX
	XOR	CL,CL
	NXTI			;jump to displacement value instruction
DJNZ1	EQU	$
	INC	SI		;bump program counter (SI) over displacement
	NXTI			;go to next instruction in line
DJNZ	ENDP
;
	ORG	11H*PG
LXID	PROC	NEAR		;11H - LXI D,D16	LD DE,D16
;				------------------------------------------
	XCHG	DX,AX		;Save AX(PSW) in DX
	LODSW			;load immediate word in AX
	XCHG	DX,AX		;put result in DE(DX) and restore AX
	NXTI
LXID	ENDP
;
	ORG	12H*PG
STAXD	PROC	NEAR		;12H - STAX D		LD (DE),A
;				------------------------------------------
	XCHG	DX,SI		;exchange DE(DX) with PC(SI)
	MOV	[SI],AL		;store A(AL) at location in DE(now SI)
	XCHG	DX,SI		;exchange DE and PC as they were
	NXTI
STAXD	ENDP
;
	ORG	13H*PG
INXD	PROC	NEAR		;13H - INX D		INC DE
;				------------------------------------------
	INC	DX		;increment DE(DX)
	NXTI
INXD	ENDP
;
	ORG	14H*PG
INRD	PROC	NEAR		;14H - INR D		INC D
;				------------------------------------------
	SAHF			;store AH in flags
	INC	DH		;increment D(DH)
	LAHF			;load flags from AH
	JO	INRD1		;jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
INRD1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
INRD	ENDP
;
	ORG	15H*PG
DCRD	PROC	NEAR		;15H - DCR D		DEC D
;				------------------------------------------
	SAHF			;store AH in flags
	DEC	DH		;decrement D(DH)
	LAHF			;load flags from AH
	JO	DCRD1		;jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
DCRD1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
DCRD	ENDP
;
	ORG	16H*PG
MVID	PROC	NEAR		;16H - MVI D,D8		LD D,D8
;				-----------------------------------------
	MOV	DH,[SI]		;load D(DH) with immediate data
	INC	SI		;bump program counter past immediate data 		
	NXTI
MVID	ENDP
;
	ORG	17H*PG
RAL	PROC	NEAR		;17H - RAL		RLA
;				-----------------------------------------
	SAHF			;store AH in flags
	RCL	AL,1		;rotate A(AL) left through carry
	LAHF			;load AH from flags
	AND	AH,0EDH		;clear aux carry and subtract flags
	NXTI
RAL	ENDP
;
	ORG	18H*PG
JR	PROC	NEAR		;18H - (none)		JR dd
;				-----------------------------------------
	XCHG	CX,AX		;save PSW(AX) in CX
	LODSB			;get displacement byte dd in AX
	CBW			;convert byte to word
	ADD	SI,AX		;add displacement to program counter(SI)
	XCHG	CX,AX		;restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;unconditional jump to PC+displacement
JR	ENDP
;
	ORG	19H*PG
DADD	PROC	NEAR		;19H - DAD D		ADD HL,DE
;				------------------------------------------
	AND	AH,0FCH		;clear subtract and carry flags
	ADD	BX,DX		;add DE(DX) to HL(BX)
	JB	DADD1		;jump if below (to set carry flag)
	NXTI
DADD1	EQU	$
	INC	AH		;set carry flag
	NXTI
DADD	ENDP
;
	ORG	1AH*PG
LDAXD	PROC	NEAR		;1AH - LDAX D		LD A,(DE)
;				-----------------------------------------
	XCHG	DX,DI		;exchange BC(DI) and DE(DX)
	MOV	AL,[DI]		;load A(AL) from DE(now in DI)
	XCHG 	DX,DI		;restore BC(DI) and DE(DX)
	NXTI
LDAXD	ENDP
;
	ORG	1BH*PG
DCXD	PROC	NEAR		;1BH - DCX D		DEC DE
;				----------------------------------------
	DEC	DX		;decrement DE(DX)
	NXTI
DCXD	ENDP
;
	ORG	1CH*PG
INRE	PROC	NEAR		;1CH - INR E		INC E
;				----------------------------------------
	SAHF			;store AH in flags
	INC	DL		;increment E(DL)
	LAHF			;load AH from flags
	JO	INRE1		;jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
INRE1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
INRE	ENDP
;
	ORG	1DH*PG
DCRE	PROC	NEAR		;1DH - DCR E		DEC E
;				---------------------------------------
	SAHF			;store AH in flags
	DEC	DL		;decrement E(DL)
	LAHF			;load AH from flags
	JO	DCRE1		;jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
DCRE1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
DCRE	ENDP
;
	ORG	1EH*PG
MVIE	PROC	NEAR		;1EH - MVI E,D8		LD E,D8
;				----------------------------------------
	MOV	DL,[SI]		;load D(DL) from immediate data
	INC	SI
	NXTI
MVIE	ENDP
;
	ORG	1FH*PG
RAR	PROC	NEAR		;1FH - RAR		RRA
;				---------------------------------------
	SAHF			;store AH in flags
	RCR	AL,1		;rotate A(AL) right through carry
	LAHF			;load AH from flags
	AND	AH,0EDH		;clear aux carry and subtract flags
	NXTI
RAR	ENDP
;
	ORG	20H*PG
JRNZ	PROC	NEAR		;20H - (none)		JR NZ,dd
;				---------------------------------------
	SAHF			;store AH in flags
	JZ	JRNZ1		;if zero, just to to next instruction
;				 if non-zero, jump to PC + dd displacement
	XCHG	CX,AX		;save PSW(AX) in CX
	LODSB			;get jump displacement byte
	CBW			;convert byte to word
	ADD	SI,AX		;add displacement to program counter(SI)
	XCHG	CX,AX		;restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;jump to PC + displacement
JRNZ1	EQU	$
	INC	SI		;bump PC(SI) over displacement byte
	NXTI			;continue with next instruction
JRNZ	ENDP
;
	ORG	21H*PG
LXIH	PROC	NEAR		;21H - LXI H,D16	LD DE,D16
;				---------------------------------------
	XCHG	BX,AX		;save PSW(AX) in HL(BX)
	LODSW			;load immediate word in AX
	XCHG	BX,AX		;immed word to HL(BX), saved PSW to AX
	NXTI
LXIH	ENDP
;
	ORG	22H*PG
SHLD	PROC	NEAR		;22H - SHLD Addr	LD (Addr),HL
;				---------------------------------------
	XCHG	CX,AX		;save PSW(AX) in CX
	LODSW			;load address in AX
	XCHG	DI,AX		;save BC(DI) in AX, address to DI
	MOV	[DI],BX		;store HL(BX) at address location
	XCHG	DI,AX		;restore BC(DI) from AX
	XCHG	CX,AX		;restore PSW(AX) from CX
	XOR	CL,CL
	NXTI
SHLD	ENDP
;
	ORG	23H*PG
INXH	PROC	NEAR		;23H - INX H		INC HL
;				---------------------------------------
	INC	BX		;increment HL(BX)
	NXTI
INXH	ENDP
;
	ORG	24H*PG
INRH	PROC	NEAR		;24H - INR H		INC H
;				--------------------------------------
	SAHF			;store AH in flags
	INC	BH		;increment H(BH)
	LAHF			;load AH from flags
	JO	INRH1		;jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
INRH1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
INRH	ENDP
;
	ORG	25H*PG
DCRH	PROC	NEAR		;25H - DCR H		DEC H
;				--------------------------------------
	SAHF			;store AH in flags
	DEC	BH		;decrement H(BH)
	LAHF			;load AH from flags
	JO	DCRH1		;jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
DCRH1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
DCRH	ENDP
;
	ORG	26H*PG
MVIH	PROC	NEAR		;26H - MVI H,D8		LD H,D8
;				---------------------------------------
	MOV	BH,[SI]		;load H(BH) from immediate byte
	INC	SI		;bump program counter past immediate data
	NXTI
MVIH	ENDP
;
	ORG	27H*PG
DAA	PROC	NEAR		;27H - DAA		DAA
;				---------------------------------------
	TEST	AH,02H		;test subtract flag
	JNZ	DAA1		;if subtract flag set do DAS
	SAHF			;store AH in flags
	DAA			;Do decimal adjust for addition
	LAHF			;load AH from flags
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
DAA1	EQU	$
	SAHF			;store AH in flags
	DAS			;Do decimal adjust for subtraction
	LAHF			;load AH from flags
	OR	AH,002H		;set subtract flag
	NXTI
DAA	ENDP
;
	ORG	28H*PG
JRZ	PROC	NEAR		;28H - (none)		JR Z,dd
;				----------------------------------------
	SAHF			;Store AH in flags
	JNZ	JRZ1		;if non-zero we do not relative jump
	XCHG	CX,AX		;save PSW(AX) in CX
	LODSB			;pick up immediate displacement byte
	CBW			;convert byte to word
	ADD	SI,AX		;add displacement to program counter(SI)
	XCHG	CX,AX		;restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;do relative jump PC+displacement
JRZ1	EQU	$
	INC	SI		;bump program counter over displacement byte
	NXTI			;continue with next instruction
JRZ	ENDP
;
	ORG	29H*PG
DADH	PROC	NEAR		;29H - DAD H		ADD HL,HL
;				----------------------------------------
	AND	AH,0FCH		;clear subtract and carry flags
	ADD	BX,BX		;add HL(BX) to HL(BX)
	JB	DADH1		;jump if below (to set carry flag)
	NXTI
DADH1	EQU	$
	INC	AH		;set the carry flag
	NXTI
DADH	ENDP
;
	ORG	2AH*PG
LHLD	PROC	NEAR		;2AH - LHLD Addr	LD HL,(Addr)
;				----------------------------------------
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;get Addr in AX
	XCHG	DI,AX		;swap BC(DI) with Addr in DI
	MOV	BX,[DI]		;load HL(BX) from contents of Addr
	XCHG	DI,AX		;restore BC(DI) from AX
	XCHG	CX,AX		;restore PSW(AX) from CX
	XOR	CL,CL
	NXTI
LHLD	ENDP
;
	ORG	2BH*PG
DCXH	PROC	NEAR		;2BH - DCX H		DEC HL
;				-----------------------------------------
	DEC	BX		;Decrement HL(BX)
	NXTI
DCXH	ENDP
;
	ORG	2CH*PG
INRL	PROC	NEAR		;2CH - INR L		INC L
;				----------------------------------------
	SAHF			;Store AH in flags
	INC	BL		;Increment L(BL)
	LAHF			;Load AH from flags
	JO	INRL1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
INRL1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
INRL	ENDP
;
	ORG	2DH*PG
DCRL	PROC	NEAR		;2DH - DCR L		DEC L
;				-----------------------------------------
	SAHF			;Store AH in flags
	DEC	BL		;Decrement L(BL)
	LAHF			;Load AH from flags
	JO	DCRL1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
DCRL1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
DCRL	ENDP
;
	ORG	2EH*PG
MVIL	PROC	NEAR		;2EH - MVI L,D8		LD L,D8
;				-----------------------------------------
	MOV	BL,[SI]		;Load L(BL) from immediate data byte
	INC	SI		;bump program counter over immed date byte
	NXTI
MVIL	ENDP
;
	ORG	2FH*PG
CMA	PROC	NEAR		;2FH - CMA		CPL 
;				-----------------------------------------
	XOR	AL,0FFH		;One's complement A(AL)
	OR	AH,012H		;Set the aux carry and subtract flags
	NXTI
CMA	ENDP
;
	ORG	30H*PG
JRNC	PROC	NEAR		;30H - (none)		JR NC,dd
;				----------------------------------------
	SAHF			;Store AH in flags.
	JB	JRNC1		;No relative jump if carry set.
	XCHG	CX,AX		;save PSW(AX) in CX
	LODSB			;load displacement byte in AL
	CBW			;convert byte to word
	ADD	SI,AX		;add displacement to program counter(SI)
	XCHG	CX,AX		;restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;do the relative jump PC+displacement
JRNC1	EQU	$
	INC	SI		;bump program counter over displacement byte
	NXTI			;continue with next instruction
JRNC	ENDP
;
	ORG	31H*PG
LXISP	PROC	NEAR		;31H - LXI SP,D16	LD SP,D16
;				-----------------------------------------
	XCHG	BP,AX		;Save PSW(AX) in BP(SP)
	LODSW			;Load AX with immediate word
	XCHG	BP,AX		;Immed word to SP(BP), restore AX
	NXTI
LXISP	ENDP
;
	ORG	32H*PG
STA	PROC	NEAR		;32H - STA Addr		LD (Addr),A
;				----------------------------------------
	XCHG	CX,AX		;Save PSW(AX) in CX - A(AL) in CL
	LODSW			;Load AX with Addr
	XCHG	DI,AX		;Save BC(DI) in AX, Addr to DI
	MOV	[DI],CL		;Store A(CL) in Addr location
	XCHG	DI,AX		;Restore BC(DI) from AX
	XCHG	CX,AX		;Restore PSW(AX) from CX
	XOR	CL,CL
	NXTI
STA	ENDP
;
	ORG	33H*PG
INXSP	PROC	NEAR		;33H - INX SP		INC SP
;				---------------------------------------
	INC	BP		;Increment SP(BP)
	NXTI
INXSP	ENDP
;
	ORG	34H*PG
INRM	PROC	NEAR		;34H - INR M		INC (HL)
;				---------------------------------------
	SAHF			;Store AH in flags.
	INC	BYTE PTR [BX]	;Increment byte pointed to by HL(BX)
	LAHF			;Load AH from flags.
	JO	INRM1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
INRM1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set the parity/overflow flag
	NXTI
INRM	ENDP
;
	ORG	35H*PG
DCRM	PROC	NEAR		;35H - DCR M		DEC (HL)
;				----------------------------------------
	SAHF			;Store AH in flags.
	DEC	BYTE PTR [BX]	;Decrement byte pointed to by HL(BX)
	LAHF			;Load AH from flags.
	JO	DCRM1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set the subtract flag
	NXTI
DCRM1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
DCRM	ENDP
;
	ORG	36H*PG
MVIM	PROC	NEAR		;36H - MVI M,D8		LD (HL),D8
;				-----------------------------------------
	MOV	CH,[SI]		;Load the immediate byte in CH
	INC	SI		;bump program counter
	MOV	[BX],CH		;store immediate byte to location in HL(BX)
	NXTI
MVIM	ENDP
;
	ORG	37H*PG
STC	PROC	NEAR		;37H - STC		SCF
;				----------------------------------------
	AND	AH,0EDH		;clear aux carry and subtract flags
	OR	AH,001H		;set the carry flag
	NXTI
STC	ENDP
;
	ORG	38H*PG
JRC	PROC	NEAR		;38H - (none)		JR C,dd
;				----------------------------------------
	SAHF			;Store AH in flags.
	JNB	JRC1		;no relative jump if no carry
	XCHG	CX,AX		;save PSW(AX) in CX
	LODSB			;get the displacement byte in AL
	CBW			;convert byte to word
	ADD	SI,AX		;add displacement to program counter(SI)
	XCHG	CX,AX		;restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;do the relative jump to PC+displacement
JRC1	EQU	$
	INC	SI		;bump program counter over displacement byte
	NXTI			;continue with next instruction
JRC	ENDP
;
	ORG	39H*PG
DADSP	PROC	NEAR		;39H - DAD SP		ADD HL,SP
;				-----------------------------------------
	AND	AH,0FCH		;clear subtract and carry flags
	ADD	BX,BP		;add the SP(BP) to HL(BX)
	JB	DADSP1		;jump on carry
	NXTI
DADSP1	EQU	$
	INC	AH		;set the carry flag
	NXTI
DADSP	ENDP
;
	ORG	3AH*PG
LDA	PROC	NEAR		;3AH - LDA Addr		LD A,(Addr)
;				----------------------------------------
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;load Addr in AX
	XCHG	DI,AX		;save BC(DI) in AX, get Addr in DI
	MOV	CL,[DI]		;Load byte in CL from location Addr
	XCHG 	DI,AX		;Restore BC(DI) from AX
	XCHG	CX,AX		;Restore flags in AH, byte read in AL(A)
	XOR	CL,CL
	NXTI
LDA	ENDP
;
	ORG	3BH*PG
DCXSP	PROC	NEAR		;3BH - DCX SP		DEC SP
;				----------------------------------------
	DEC	BP		;Decrement SP(BP)
	NXTI
DCXSP	ENDP
;
	ORG	3CH*PG
INRA	PROC	NEAR		;3CH - INR A		INC A
;				----------------------------------------
	SAHF			;Store AH in flags.
	INC	AL		;Increment A(AL)
	LAHF			;Load AH from flags.
	JO	INRA1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
INRA1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set the parity/overflow flag
	NXTI
INRA	ENDP
;
	ORG	3DH*PG
DCRA	PROC	NEAR		;3DH - DCR A		DEC A
;				-----------------------------------------
	SAHF			;Store AH in flags.
	DEC	AL		;Decrement A(AL)
	LAHF			;Load AH from flags.
	JO	DCRA1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set the subtract flag
	NXTI
DCRA1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
DCRA	ENDP
;
	ORG	3EH*PG
MVIA	PROC	NEAR		;3EH - MVI A,D8		LD A,D8
;				----------------------------------------
	MOV	AL,[SI]		;Load A(AL) from immediate data byte
	INC	SI		;bump the program counter
	NXTI
MVIA	ENDP
;
	ORG	3FH*PG
CMC	PROC	NEAR		;3FH - CMC		CCF
;				----------------------------------------
	AND	AH,0EDH		;clear aux carry and subtract flags
	XOR	AH,001H		;complement (reverse) the carry flag
	NXTI
CMC	ENDP
;
	ORG	40H*PG
MOVBB	PROC	NEAR		;40H - MOV B,B		LD B,B
;				----------------------------------------
	XCHG	CX,DI		;put BC(DI) in CX
	MOV	CH,CH		;move B(CH) to B(CH)
	XCHG	CX,DI		;put result back in BC(DI)
	NXTI
MOVBB	ENDP
;
	ORG	41H*PG
MOVBC	PROC	NEAR		;41H - MOV B,C		LD B,C
;				---------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AH,AL		;move C(AL) to B(AH)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVBC	ENDP
;
	ORG	42H*PG
MOVBD	PROC	NEAR		;42H - MOV B,D		LD B,D
;				---------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AH,DH		;move D(DH) to B(AH)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVBD	ENDP
;
	ORG	43H*PG
MOVBE	PROC	NEAR		;43H - MOV B,E		LD B,E
;				--------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AH,DL		;move E(DL) to B(AH)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVBE	ENDP
;
	ORG	44H*PG
MOVBH	PROC	NEAR		;44H - MOV B,H		LD B,H
;				--------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AH,BH		;move H(BH) to B(AH)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVBH	ENDP
;
	ORG	45H*PG
MOVBL	PROC	NEAR		;45H - MOV B,L		LD B,L
;				--------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AH,BL		;move L(BL) to B(AH)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVBL	ENDP
;
	ORG	46H*PG
MOVBM	PROC	NEAR		;46H - MOV B,M		LD B,(HL)
;				---------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AH,[BX]		;move byte pointed to by HL to B(AH)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVBM	ENDP
;
	ORG	47H*PG
MOVBA	PROC	NEAR		;47H - MOV B,A		LD B,A
;				--------------------------------------
	XCHG	CX,DI		;put BC(DI) in CX
	MOV	CH,AL		;move A(AL) to B(CH)
	XCHG	CX,DI		;put result in BC(DI)
	NXTI
MOVBA	ENDP
;
	ORG	48H*PG
MOVCB	PROC	NEAR		;48H - MOV C,B		LD C,B
;				--------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AL,AH		;move B(AH) to C(AL)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVCB	ENDP
;
	ORG	49H*PG
MOVCC	PROC	NEAR		;49H - MOV C,C		LD C,C
;				---------------------------------------
	XCHG	CX,DI		;put BC(DI) in CX
	MOV	CL,CL		;move C(CL) to C(CL)
	XCHG	CX,DI		;put result in BC(DI)
	NXTI
MOVCC	ENDP
;
	ORG	4AH*PG
MOVCD	PROC	NEAR		;4AH - MOV C,D		LD C,D
;				----------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AL,DH		;move D(DH) to C(AL)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVCD	ENDP
;
	ORG	4BH*PG
MOVCE	PROC	NEAR		;4BH - MOV C,E		LD C,E
;				----------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AL,DL		;move E(DL) to C(AL)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVCE	ENDP
;
	ORG	4CH*PG
MOVCH	PROC	NEAR		;4CH - MOV C,H		LD C,H
;				----------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AL,BH		;move H(BH) to C(AL)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVCH	ENDP
;
	ORG	4DH*PG
MOVCL	PROC	NEAR		;4DH - MOV C,L		LD C,L
;				----------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AL,BL		;move L(BL) to C(AL)
	XCHG	DI,AX		;put result in BC(DI), restore AX
	NXTI
MOVCL	ENDP
;
	ORG	4EH*PG
MOVCM	PROC	NEAR		;4EH - MOV C,M		LD C,(HL)
;				----------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	AL,[BX]		;load C(AL) from byte pointed to by HL(BX)
	XCHG	DI,AX		;put result in DI(BC), restore AX
	NXTI
MOVCM	ENDP
;
	ORG	4FH*PG
MOVCA	PROC	NEAR		;4FH - MOV C,A		LD C,A
;				----------------------------------------
	XCHG	CX,DI		;put BC(DI) in CX
	MOV	CL,AL		;move A(AL) to C(CL)
	XCHG	CX,DI		;put result in BC(DI)
	NXTI
MOVCA	ENDP
;
	ORG	50H*PG
MOVDB	PROC	NEAR		;50H - MOV D,B		LD D,B
;				-----------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	DH,AH		;move B(AH) to D(DH)
	XCHG	DI,AX		;restore BC(DI) and PSW(AX)
	NXTI
MOVDB	ENDP
;
	ORG	51H*PG
MOVDC	PROC	NEAR		;51H - MOV D,C		LD D,C
;				-----------------------------------------
	XCHG	DI,AX		;put BC(DI) in AX, save PSW(AX) in DI
	MOV	DH,AL		;move C(AL) to D(DH)
	XCHG	DI,AX		;restore BC(DI) and PSW(AX)
	NXTI
MOVDC	ENDP
;
	ORG	52H*PG
MOVDD	PROC	NEAR		;52H - MOV D,D		LD D,D
;				----------------------------------------
	MOV	DH,DH		;Move D(DH) to D(DH)
	NXTI
MOVDD	ENDP
;
	ORG	53H*PG
MOVDE	PROC	NEAR		;53H - MOV D,E		LD D,E
;				----------------------------------------
	MOV	DH,DL		;Move E(DL) to D(DH)
	NXTI
MOVDE	ENDP
;
	ORG	54H*PG
MOVDH	PROC	NEAR		;54H - MOV D,H		LD D,H
;				----------------------------------------
	MOV	DH,BH		;Move H(BH) to D(DH)
	NXTI
MOVDH	ENDP
;
	ORG	55H*PG
MOVDL	PROC	NEAR		;55H - MOV D,L		LD D,L
;				----------------------------------------
	MOV	DH,BL		;Move L(BL) to D(DH)
	NXTI
MOVDL	ENDP
;
	ORG	56H*PG
MOVDM	PROC	NEAR		;56H - MOV D,M		LD D,(HL)
;				----------------------------------------
	MOV	DH,[BX]		;Move byte pointed to by HL(BX) to D(DH)
	NXTI
MOVDM	ENDP
;
	ORG	57H*PG
MOVDA	PROC	NEAR		;57H - MOV D,A		LD D,A
;				----------------------------------------
	MOV	DH,AL		;Move A(AL) to D(DH)
	NXTI
MOVDA	ENDP
;
	ORG	58H*PG
MOVEB	PROC	NEAR		;58H - MOV E,B		LD E,B
;				----------------------------------------
	XCHG	DI,AX		;Put BC(DI) in AX, save PSW(AX) in DI
	MOV	DL,AH		;Move B(AH) to E(DL)
	XCHG	DI,AX		;Restore BC(DI) and PSW(AX)
	NXTI
MOVEB	ENDP
;
	ORG	59H*PG
MOVEC	PROC	NEAR		;59H - MOV E,C		LD E,C
;				----------------------------------------
	XCHG	DI,AX		;Put BC(DI) in AX, save PSW(AX) in DI
	MOV	DL,AL		;Move C(AL) to E(DL)
	XCHG	DI,AX		;Restore BC(DI) and PSW(AX)
	NXTI
MOVEC	ENDP
;
	ORG	5AH*PG
MOVED	PROC	NEAR		;5AH - MOV E,D		LD E,D
;				----------------------------------------
	MOV	DL,DH		;Move D(DH) to E(DL)
	NXTI
MOVED	ENDP
;
	ORG	5BH*PG
MOVEE	PROC	NEAR		;5BH - MOV E,E		LD E,E
;				---------------------------------------
	MOV	DL,DL		;Move E(DL) to E(DL)
	NXTI
MOVEE	ENDP
;
	ORG	5CH*PG
MOVEH	PROC	NEAR		;5CH - MOV E,H		LD E,H
;				---------------------------------------
	MOV	DL,BH		;Move H(BH) to E(DL)
	NXTI
MOVEH	ENDP
;
	ORG	5DH*PG
MOVEL	PROC	NEAR		;5DH - MOV E,L		LD E,L
;				---------------------------------------
	MOV	DL,BL		;Move L(BL) to E(DL)
	NXTI
MOVEL	ENDP
;
	ORG	5EH*PG
MOVEM	PROC	NEAR		;5EH - MOV E,M		LD E,(HL)
;				----------------------------------------
	MOV	DL,[BX]		;Move byte pointed to by HL(BX) to E(DL)
	NXTI
MOVEM	ENDP
;
	ORG	5FH*PG
MOVEA	PROC	NEAR		;5FH - MOV E,A		LD E,A
;				----------------------------------------
	MOV	DL,AL		;Move A(AL) to E(DL)
	NXTI
MOVEA	ENDP
;
	ORG	60H*PG
MOVHB	PROC	NEAR		;60H - MOV H,B		LD H,B
;				-----------------------------------------
	XCHG	DI,AX		;Put BC(DI) in AX, save PSW(AX) in DI
	MOV	BH,AH		;Mov B(AH) to H(BH)
	XCHG	DI,AX		;Restore BC(DI) and PSW(AX)
	NXTI
MOVHB	ENDP
;
	ORG	61H*PG
MOVHC	PROC	NEAR		;61H - MOV H,C		LD H,C
;				-----------------------------------------
	XCHG	DI,AX		;Put BC(DI) in AX, save PSW(AX) in DI
	MOV	BH,AL		;Move C(AL) to H(BH)
	XCHG	DI,AX		;Restore BC(DI) and PSW(AX)
	NXTI
MOVHC	ENDP
;
	ORG	62H*PG
MOVHD	PROC	NEAR		;62H - MOV H,D		LD H,D
;				-----------------------------------------
	MOV	BH,DH		;Move D(DH) to H(BH)
	NXTI
MOVHD	ENDP
;
	ORG	63H*PG
MOVHE	PROC	NEAR		;63H - MOV H,E		LD H,E
;				-----------------------------------------
	MOV	BH,DL		;Move E(DL) to H(BH)
	NXTI
MOVHE	ENDP
;
	ORG	64H*PG
MOVHH	PROC	NEAR		;64H - MOV H,H		LD H,H
;				-----------------------------------------
	MOV	BH,BH		;Move H(BH) to H(BH)
	NXTI
MOVHH	ENDP
;
	ORG	65H*PG
MOVHL	PROC	NEAR		;65H - MOV H,L		LD H,L
;				----------------------------------------
	MOV	BH,BL		;Move L(BL) to H(BH)
	NXTI
MOVHL	ENDP
;
	ORG	66H*PG
MOVHM	PROC	NEAR		;66H - MOV H,M		LD H,(HL)
;				----------------------------------------
	MOV	BH,[BX]		;Move byte pointed to by HL(BX) into H(BH)
	NXTI
MOVHM	ENDP
;
	ORG	67H*PG
MOVHA	PROC	NEAR		;67H - MOV H,A		LD H,A
;				----------------------------------------
	MOV	BH,AL		;Mov A(AL) to H(BH)
	NXTI
MOVHA	ENDP
;
	ORG	68H*PG
MOVLB	PROC	NEAR		;68H - MOV L,B		LD L,B
;				---------------------------------------
	XCHG	DI,AX		;Put BC(DI) in AX, save PSW(AX) in DI
	MOV	BL,AH		;Move B(AH) to L(BL)
	XCHG	DI,AX		;Restore BC(DI) and PSW(AX)
	NXTI
MOVLB	ENDP
;
	ORG	69H*PG
MOVLC	PROC	NEAR		;69H - MOV L,C		LD L,C
;				--------------------------------------
	XCHG	DI,AX		;Put BC(DI) in AX, save PSW(AX) in DI
	MOV	BL,AL		;Move C(AL) to L(BL)
	XCHG	DI,AX		;Restore BC(DI) and PSW(AX)
	NXTI
MOVLC	ENDP
;
	ORG 	6AH*PG
MOVLD	PROC	NEAR		;6AH - MOV L,D		LD L,D
;				--------------------------------------
	MOV	BL,DH		;Move D(DH) to L(BL)
	NXTI
MOVLD	ENDP
;
	ORG	6BH*PG
MOVLE	PROC	NEAR		;6BH - MOV L,E		LD L,E
;				---------------------------------------
	MOV	BL,DL		;Move E(DL) to L(BL)
	NXTI
MOVLE	ENDP
;
	ORG	6CH*PG
MOVLH	PROC	NEAR		;6CH - MOV L,H		LD L,H
;				---------------------------------------
	MOV	BL,BH		;Move H(BH) to L(BL)
	NXTI
MOVLH	ENDP
;
	ORG	6DH*PG
MOVLL	PROC	NEAR		;6DH - MOV L,L		LD L,L
;				---------------------------------------
	MOV	BL,BL		;Move L(BL) to L(BL)
	NXTI
MOVLL	ENDP
;
	ORG 	6EH*PG
MOVLM	PROC	NEAR		;6EH - MOV L,M		LD L,(HL)
;				----------------------------------------
	MOV	BL,[BX]		;Move byte pointed to by HL(BX) to L(BL)
	NXTI
MOVLM	ENDP
;
	ORG	6FH*PG
MOVLA	PROC	NEAR		;6FH - MOV L,A		LD L,A
;				---------------------------------------
	MOV	BL,AL		;Move A(AL) to L(BL)
	NXTI
MOVLA	ENDP
;
	ORG	70H*PG
MOVMB	PROC	NEAR		;70H - MOV M,B		LD (HL),B
;				---------------------------------------
	XCHG	DI,AX		;Put BC(DI) in AX, save PSW(AX) in DI
	MOV	[BX],AH		;Move B(AH) to byte pointed to by HL(BX)
	XCHG	DI,AX		;Restore BC(DI) and PSW(AX)
	NXTI
MOVMB	ENDP
;
	ORG	71H*PG
MOVMC	PROC	NEAR		;71H - MOV M,C		LD (HL),C
;				---------------------------------------
	XCHG	DI,AX		;Put BC(DI) in AX, save PSW(AX) in DI
	MOV	[BX],AL		;Move C(AL) to byte pointed to by HL(BX)
	XCHG	DI,AX		;Restore BC(DI) and PSW(AX)
	NXTI
MOVMC	ENDP
;
	ORG	72H*PG
MOVMD	PROC	NEAR		;72H - MOV M,D		LD (HL),D
;				---------------------------------------
	MOV	[BX],DH		;Move D(DH) to byte pointed to by HL(BX)
	NXTI
MOVMD	ENDP
;
	ORG	73H*PG
MOVME	PROC	NEAR		;73H - MOV M,E		LD (HL),E
;				---------------------------------------
	MOV	[BX],DL		;Move E(DL) to byte pointed to by HL(BX)
	NXTI
MOVME	ENDP
;
	ORG	74H*PG
MOVMH	PROC	NEAR		;74H - MOV M,H		LD (HL),H
;				---------------------------------------
	MOV	[BX],BH		;Move H(BH) to byte pointed to by HL(BX)
	NXTI
MOVMH	ENDP
;
	ORG	75H*PG
MOVML	PROC	NEAR		;75H - MOV M,L		LD (HL),L
;				---------------------------------------
	MOV	[BX],BL		;Move L(BL) to byte pointed to by HL(BX)
	NXTI
MOVML	ENDP
;
	ORG	76H*PG
HLT	PROC	NEAR		;76H - HLT		HALT
;				---------------------------------------
	HLT			;halt here until an interrupt occurs
	NXTI
HLT	ENDP
;
	ORG	77H*PG
MOVMA	PROC	NEAR		;77H - MOV M,A		LD (HL),A
;				---------------------------------------
	MOV	[BX],AL		;Move A(AL) to byte pointed to by HL(BX)
	NXTI
MOVMA	ENDP
;
	ORG	78H*PG
MOVAB	PROC	NEAR		;78H - MOV A,B		LD A,B
;				-----------------------------------------
	XCHG	CX,DI		;Put BC(DI) in CX
	MOV	AL,CH		;Move B(CH) to A(AL)
	XCHG	CX,DI		;Restore BC(DI) from CX
	NXTI
MOVAB	ENDP
;
	ORG	79H*PG
MOVAC	PROC	NEAR		;79H - MOV A,C		LD A,C
;				-----------------------------------------
	XCHG	CX,DI		;Put BC(DI) in CX
	MOV	AL,CL		;Move C(CL) to A(AL)
	XCHG	CX,DI		;Restore BC(DI) from CX
	NXTI
MOVAC	ENDP
;
	ORG	7AH*PG
MOVAD	PROC	NEAR		;7AH - MOV A,D		LD A,D
;				----------------------------------------
	MOV	AL,DH		;Move D(DH) to A(AL)
	NXTI
MOVAD	ENDP
;
	ORG	7BH*PG
MOVAE	PROC	NEAR		;7BH - MOV A,E		LD A,E
;				----------------------------------------
	MOV	AL,DL		;Move E(DL) to A(AL)
	NXTI
MOVAE	ENDP
;
	ORG	7CH*PG
MOVAH	PROC	NEAR		;7CH - MOV A,H		LD A,H
;				-----------------------------------------
	MOV	AL,BH		;Move H(BH) to A(AL)
	NXTI
MOVAH	ENDP
;
	ORG	7DH*PG
MOVAL	PROC	NEAR		;7DH - MOV A,L		LD A,L
;				-----------------------------------------
	MOV	AL,BL		;Move L(BL) to A(AL)
	NXTI
MOVAL	ENDP
;
	ORG	7EH*PG
MOVAM	PROC	NEAR		;7EH - MOV A,M		LD A,(HL)
;				-----------------------------------------
	MOV	AL,[BX]		;Move byte pointed to by HL(BX) to A(AL)
	NXTI
MOVAM	ENDP
;
	ORG 	7FH*PG
MOVAA	PROC	NEAR		;7FH - MOV A,A		LD A,A
;				-----------------------------------------
	MOV	AL,AL		;Move A(AL) to A(AL)
	NXTI
MOVAA	ENDP
;
