/* digits.c
 * Copyright (C) 2000, Tsurishaddai Williamson, tsuri@earthlink.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**********************************************************************/

#include <ctype.h>
#include "digits.h"

/* UnsignedToBCD() converts an unsigned value to BinaryCodedDecimal. */
unsigned UnsignedToBCD(unsigned value)
{

	return ((value / 10) << 4) + (value % 10);

}

/* IsChar() tests if a value fits into one byte. */
int IsChar(long value)
{

	if (value & ~0xFF)
		if ((value > 0) || (value < -0x7F))
			goto error;

	return 1;

error:
	return 0;

}

/* IsShort() tests if a value fits into two bytes. */
int IsShort(long value)
{

	if (value & ~0xFFFF)
		if ((value > 0) || (value < -0x7FFF))
			goto error;

	return 1;

error:
	return 0;

}

/* DigitsToLong() converts a digit string to a long value. */
int DigitsToLong(const char *number, long *value, long radix)
{
	const char *n = number;
	long digit;

	*value = 0;
	while (*n != 0) {
		if ((*n >= '0') && (*n <= '9'))
			digit = *n - '0';
		else if ((*n >= 'A') && (*n <= 'F'))
			digit = 10 + *n - 'A';
		else if ((*n >= 'a') && (*n <= 'f'))
			digit = 10 + *n - 'a';
		else
			break;
		if (digit >= radix)
			break;
		*value = (*value * radix) + digit;
		n++;
	}

	/* All done, no error, return the number of digits. */
	return n - number;

	/* Return 0 if there was an error. */
error:
	return 0;

}

/* DigitsToInt() converts a digit string to an int value. */
int DigitsToInt(const char *number, int *valuePtr, long radix)
{
	long value;
	int n;

	/* Convert the digit string to a value. */
	if ((n = DigitsToLong(number, &value, radix)) == 0)
		goto error;

	/* Error if overflow. */
	if ((sizeof(int) == sizeof(short)) && !IsShort(value))
		goto error;

	/* Set the int value. */
	*valuePtr = value;

	/* All done, no error, return the number of characters used. */
	return n;

	/* Return 0 if there was an error. */
error:
	return 0;

}

/* DigitsToShort() converts a digit string to a short value. */
int DigitsToShort(const char *digits, short *valuePtr, long radix)
{
	long value;
	int n;

	/* Convert the digit string to a value. */
	if ((n = DigitsToLong(digits, &value, radix)) == 0)
		goto error;

	/* Error if overflow. */
	if (!IsShort(value))
		goto error;

	/* Set the short value. */
	*valuePtr = value;

	/* All done, no error, return the number of digits. */
	return n;

	/* Return 0 if there was an error. */
error:
	return 0;

}

/* DigitsToChar() converts a digit string to a char value. */
int DigitsToChar(const char *digits, char *valuePtr, long radix)
{
	long value;
	int n;

	/* Convert the digit string to a value. */
	if ((n = DigitsToLong(digits, &value, radix)) == 0)
		goto error;

	/* Error if overflow. */
	if (!IsChar(value))
		goto error;

	/* Set the char value. */
	*valuePtr = value;

	/* All done, no error, return the number of digits. */
	return n;

	/* Return 0 if there was an error. */
error:
	return 0;

}

/* StringToLong() converts a number string to a long value. */
int StringToLong(const char *number, long *valuePtr)
{
	const char *nextChar = number;
	char buffer[33];
	char *n;
	long radix;
	long sign;
	long units;
	unsigned i;

	/* A number may begin with '+' or '-'. */
	if (*nextChar == '+') {
		sign = 1;
		nextChar++;
	}
	else if (*nextChar == '-') {
		sign = -1;
		nextChar++;
	}
	else
		sign = 1;

	/* A number must start with a decimal digit. */
	if (!isdigit(*nextChar))
		goto error;

	/* Hex numbers can start with '0x'. */
	radix = 0;
	if ((nextChar[0] == '0') && (toupper(nextChar[1]) == 'X')) {
		radix = 16;
		nextChar += 2;
	}

	/* Scan for all of the digits. */
	i = 0;
	n = buffer;
	while (isxdigit(*nextChar) || (*nextChar == '$')) {
		if (*nextChar != '$') {
			if (i++ >= sizeof(buffer))
				goto error;
			*n++ = toupper(*nextChar);
		}
		nextChar++;
	}
	*n = 0;
	if (i == 0)
		goto error;

	/* If the radix was not specified as '0x'... */
	if (radix == 0) {
		/* then determine the radix. */
		if ((n[-1] == 'B') && (toupper(nextChar[0]) != 'H')) {
			nextChar--;
			*--n = 0;
		}
		switch (toupper(*nextChar++)) {
		case 'B':
			radix = 2;
			break;
		case 'H':
			radix = 16;
			break;
		case 'O':
		case 'Q':
			radix = 8;
			break;
		case 'D':
			radix = 10;
			break;
		default:
			radix = 10;
			nextChar--;
			break;
		}
	}

	/* Determine the units. */
	switch (toupper(*nextChar++)) {
	case 'G':
		units = 1024 * 1024 * 1024;
		break;
	case 'M':
		units = 1024 * 1024;
		break;
	case 'K':
		units = 1024;
		break;
	default:
		units = 1;
		nextChar--;
		break;
	}

	/* Convert the digits to a value. */
	if (DigitsToLong(buffer, valuePtr, radix) == 0)
		goto error;

	/* Apply the units. */
	if (units != 1)
		*valuePtr *= units;

	/* Apply the sign. */
	if (sign < 0)
		*valuePtr = - *valuePtr;

	/* All done, no error, return the number of characters used. */
	return nextChar - number;

	/* Return 0 if there was an error. */
error:
	return 0;

}

/* StringToInt() converts a number string to an int value. */
int StringToInt(const char *number, int *valuePtr)
{
	long value;
	int n;

	/* Conver the number string to a value. */
	if ((n = StringToLong(number, &value)) == 0)
		goto error;

	/* Error if overflow. */
	if ((sizeof(int) == sizeof(short)) && !IsShort(value))
		goto error;

	/* Set the int value. */
	*valuePtr = value;

	/* All done, no error, return the number of characters used. */
	return n;

	/* Return 0 if there was an error. */
error:
	return 0;

}

/* StringToShort() converts a number string to a short value. */
int StringToShort(const char *number, short *valuePtr)
{
	long value;
	int n;

	/* Conver the number string to a value. */
	if ((n = StringToLong(number, &value)) == 0)
		goto error;

	/* Error if overflow. */
	if (!IsShort(value))
		goto error;

	/* Set the short value. */
	*valuePtr = value;

	/* All done, no error, return the number of characters used. */
	return n;

	/* Return 0 if there was an error. */
error:
	return 0;

}

/* StringToChar() converts a number string to a char value. */
int StringToChar(const char *number, char *valuePtr)
{
	long value;
	int n;

	/* Conver the number string to a value. */
	if ((n = StringToLong(number, &value)) == 0)
		goto error;

	/* Error if overflow. */
	if (!IsChar(value))
		goto error;

	/* Set the char value. */
	*valuePtr = value;

	/* All done, no error, return the number of characters used. */
	return n;

	/* Return 0 if there was an error. */
error:
	return 0;

}
