/* uSim dasm.c
 * Copyright (C) 2000, Tsurishaddai Williamson, tsuri@earthlink.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**********************************************************************/

/*
 * Code disssembly covers three situations.
 * 1) Disassembly of a single instruction in memory.
 * 2) Disassembly of the CPU State.
 * 3) Disassembly of static code in memory.
 *
 * Individual instructions are disassembled via a table indexed by the
 * operation code.  A format string controls the interpretation of any
 * operands similar to the way printf() uses a format string to control
 * the interpretation of its arguments.  The DisassembleInstruction()
 * determines the instruction size, and therefore the next instruction
 * address.  It also notes any address referenced by the instruction.
 *
 * The CPU State consists of the CPU Registers and the instruction that
 * is about to be executed as referenced by the Program Counter.  Note
 * that the Z80 CPU State is two lines long to handle the alternate
 * register set.
 *
 * Static code is disassembled by following and noting all code and data
 * references from a given starting address.  The output file contains
 * everything except dynamic indirect code and data references.
 *
 * It is possible to collect dynamic reference information to follow the
 * indirect code and data references...
 */

/**********************************************************************/

#include "stdio.h"
#include <string.h>
#include <stdlib.h>

#include "memory.h"
#include "system.h"
#include "cpu.h"
#include "dasm.h"

/**********************************************************************/

/*
 * The optab[] is indexed by the instruction.  It provides a string
 * which consists of the reference code followed by the print format
 * of the instruction.
 * See "dasm.h" for a list of the instruction reference code.
 */

#ifdef Z80

#define OPERATION(INDEX, CODE, JMP, N8080, NZ80, F8080, FZ80) \
	#JMP##NZ80,

static char * optab_CB[256] = {
#include "opcb.h"
};

static char * optab_DD[256] = {
#include "opdd.h"
};

static char * optab_DDCB[256] = {
#include "opddcb.h"
};

static char * optab_ED[256] = {
#include "oped.h"
};

static char * optab_FD[256] = {
#include "opfd.h"
};

static char * optab_FDCB[256] = {
#include "opfdcb.h"
};

static char * optab[256] = {
#include "op.h"
};

#else

#define OPERATION(INDEX, CODE, JMP, N8080, NZ80, F8080, FZ80) \
	#JMP##N8080,

static char * optab[256] = {
#include "op.h"
};

#endif

/**********************************************************************/

/* HexByte() converts a byte to two hex digits. */
/* The referenced stringPtr is advanced by two. */
static void HexByte(char **stringPtr, Byte x)
{
	static char *hex = "0123456789ABCDEF";
	*(*stringPtr)++ = hex[x >> 4];
	*(*stringPtr)++ = hex[x & 0x0F];
}

/* DisassembleInstruction() disassembles one instruction. */
void
	DisassembleInstruction(char *buffer,
	                       Word address,
	                       Byte *memory,
	                       Word *_nextAddress,
	                       char *_referenceCode,
	                       Word *_referenceAddress)
{
	char *b = buffer;
	char *f;
	WordBytes referenceAddress;
	char referenceCode;
	Word nextAddress;

	/* The next address follows this address. */
	nextAddress = address + 1;
	/* Get the format string for this instruction. */
#ifdef Z80
	switch (memory[0]) {
	case 0xCB:
		nextAddress++;
		f = optab_CB[memory[1]];
		break;
	case 0xDD:
		nextAddress++;
		if (memory[1] == 0xCB) {
			f = optab_DDCB[memory[3]];
			nextAddress += 2;
		}
		else
			f = optab_DD[memory[1]];
		break;
	case 0xED:
		nextAddress++;
		f = optab_ED[memory[1]];
		break;
	case 0xFD:
		nextAddress++;
		if (memory[1] == 0xCB) {
			f = optab_FDCB[memory[3]];
			nextAddress += 2;
		}
		else
			f = optab_FD[memory[1]];
		break;
	default:
		f = optab[memory[0]];
		break;
	}
#else
	f = optab[memory[0]];
#endif

	/* Get the reference type for this instruction. */
	referenceCode = *f++;
	referenceAddress.word = 0;

	/* Unspecified instructions are UOP's. */
	if (*f == 0)
		f = "UOP(%8)";

	/* Interpret the instruction format string. */
	while (*f) {
		if (*f != '%') {
			*b++ = *f++;
			continue;
		}
		f++;
		if (*f == '%') {
			*b++ = *f++;
			continue;
		}
		switch (*f++) {
		/* %1 -- byte value, xxNN */
		case '1':
			HexByte(&b, memory[1]);
			nextAddress += 1;
			break;
		/* %2 -- word value, xxNNNN */
		case '2':
			HexByte(&b, referenceAddress.byte.high = memory[2]);
			HexByte(&b, referenceAddress.byte.low = memory[1]);
			nextAddress += 2;
			break;
		/* %3 -- PC relative word value, xxNN */
		case '3':
			referenceAddress.word = address + (char)memory[1];
			HexByte(&b, referenceAddress.byte.high);
			HexByte(&b, referenceAddress.byte.low);
			nextAddress += 1;
			break;
		/* %4 -- word value, xxxxNNNN */
		case '4':
			HexByte(&b, referenceAddress.byte.high = memory[3]);
			HexByte(&b, referenceAddress.byte.low = memory[2]);
			nextAddress += 2;
			break;
		/* %5 -- byte value, xxxxNNxx */
		case '5':
			HexByte(&b, memory[2]);
			break;
		/* %6 -- byte value, xxxxNN or xxxxNNxx */
		case '6':
			HexByte(&b, memory[2]);
			nextAddress += 1;
			break;
		/* %7 -- byte value, xxxxxxNN */
		case '7':
			HexByte(&b, memory[3]);
			nextAddress += 1;
			break;
		/* %8 -- undefined operation, NNNNNNNN */
		case '8':
			HexByte(&b, memory[0]);
			HexByte(&b, memory[1]);
			HexByte(&b, memory[2]);
			HexByte(&b, memory[3]);
			break;
		}
	}
	*b = 0;

	/* Return nextAddress, referenceCode and referenceAddress. */
	if (_nextAddress != 0)
		*_nextAddress = nextAddress;
	if (_referenceCode != 0)
		*_referenceCode = referenceCode;
	if (_referenceAddress != 0)
		*_referenceAddress = referenceAddress.word;

}

/**********************************************************************/

/* DissasembleCpuState() disassembles the entire CPU state. */
void DissasembleCpuState(char *buffer, CpuStatePtr s)
{
	Byte memory[4];

	/* Reference up to 4 bytes of instruction. */
	memory[0] = RdByte(s->pc.word + 0);
	memory[1] = RdByte(s->pc.word + 1);
	memory[2] = RdByte(s->pc.word + 2);
	memory[3] = RdByte(s->pc.word + 3);

	/* Format the CPU Registers. */
#ifdef Z80
	sprintf(buffer,
	        "C%dZ%dM%dE%dI%dN%d "
	        "A=%02X B=%04X D=%04X H=%04X X=%04X Y=%04X S=%04X P=%04X ",
	        (s->af.byte.low & CARRY) ? 1 : 0,
	        (s->af.byte.low & ZERO) ? 1 : 0,
	        (s->af.byte.low & SIGN) ? 1 : 0,
	        (s->af.byte.low & PARITY) ? 1 : 0,
	        (s->af.byte.low & HALFCARRY) ? 1 : 0,
	        (s->af.byte.low & SUBTRACT) ? 1 : 0,
	        s->af.byte.high,
	        s->bc.word,
	        s->de.word,
	        s->hl.word,
	        s->ix.word,
	        s->iy.word,
	        s->sp.word,
	        s->pc.word);
#else
	sprintf(buffer,
	        "C%dZ%dM%dE%dI%dN%d "
	        "A=%02X B=%04X D=%04X H=%04X S=%04X P=%04X ",
	        (s->af.byte.low & CARRY) ? 1 : 0,
	        (s->af.byte.low & ZERO) ? 1 : 0,
	        (s->af.byte.low & SIGN) ? 1 : 0,
	        (s->af.byte.low & PARITY) ? 1 : 0,
	        (s->af.byte.low & HALFCARRY) ? 1 : 0,
	        (s->af.byte.low & SUBTRACT) ? 1 : 0,
	        s->af.byte.high,
	        s->bc.word,
	        s->de.word,
	        s->hl.word,
	        s->sp.word,
	        s->pc.word);
#endif

	/* Format the instruction. */
	DisassembleInstruction(strchr(buffer, 0),
	                       s->pc.word,
	                       memory,
	                       0,
	                       0,
	                       0);

#ifdef Z80
	/* Format the alternate CPU Registers. */
	sprintf(strchr(buffer, 0),
	        "\nC%dZ%dM%dE%dI%dN%d "
	        "A=%02X B=%04X D=%04X H=%04X",
	        (s->prime.af.byte.low & CARRY) ? 1 : 0,
	        (s->prime.af.byte.low & ZERO) ? 1 : 0,
	        (s->prime.af.byte.low & SIGN) ? 1 : 0,
	        (s->prime.af.byte.low & PARITY) ? 1 : 0,
	        (s->prime.af.byte.low & HALFCARRY) ? 1 : 0,
	        (s->prime.af.byte.low & SUBTRACT) ? 1 : 0,
	        s->prime.af.byte.high,
	        s->prime.bc.word,
	        s->prime.de.word,
	        s->prime.hl.word);
#endif

}

/**********************************************************************/

/* Instruction reference flags. */
#define IS_CODE  1
#define IS_DATA  2
#define IS_LABEL 4

/* SetReference() sets the reference flag for the referenced address. */
static int
	SetReference(Byte *flags,
	             Byte referenceFlag,
	             Word referenceAddress)
{
	int newReference;

	/* New reference if no previous reference. */
	newReference = ((flags[referenceAddress] & referenceFlag) == 0);

	/* Note the reference. */
	flags[referenceAddress] |= referenceFlag;

	/* Return true if this is a new reference. */
	return newReference;

}

/* DissasembleReference() notes references made by this instruction. */
static int
	DissasembleReference(Byte *flags,
	                     Word address,
	                     Byte *memory,
	                     Word *nextAddress)
{
	char buffer[256];
	Word referenceAddress;
	char referenceCode;
	int newReference;

	/* Disssemble the struction at this address. */
	/* Compute the next instruction address. */
	/* Note any reference address. */
	DisassembleInstruction(buffer,
	                       address,
	                       memory,
	                       nextAddress,
	                       &referenceCode,
	                       &referenceAddress);

	/* Note all references made by this instruction. */
	newReference = SetReference(flags, IS_CODE, address);
	switch(referenceCode) {
	case SIMPLE:
		newReference |= SetReference(flags, IS_CODE, *nextAddress);
		break;
	case JMP_ALWAYS:
		newReference |= SetReference(flags, IS_LABEL, referenceAddress);
		break;
	case JMP_SOMETIMES:
		newReference |= SetReference(flags, IS_LABEL, referenceAddress);
		newReference |= SetReference(flags, IS_CODE, *nextAddress);
		break;
	case RET_ALWAYS:
		break;
	case RET_SOMETIMES:
		newReference |= SetReference(flags, IS_CODE, *nextAddress);
		break;
	case REF_DATA:
		newReference |= SetReference(flags, IS_DATA, referenceAddress);
		newReference |= SetReference(flags, IS_CODE, *nextAddress);
		break;
	case RST00:
		newReference |= SetReference(flags, IS_LABEL, 0x0000);
		break;
	case RST08:
		newReference |= SetReference(flags, IS_LABEL, 0x0008);
		break;
	case RST10:
		newReference |= SetReference(flags, IS_LABEL, 0x0010);
		break;
	case RST18:
		newReference |= SetReference(flags, IS_LABEL, 0x0018);
		break;
	case RST20:
		newReference |= SetReference(flags, IS_LABEL, 0x0020);
		break;
	case RST28:
		newReference |= SetReference(flags, IS_LABEL, 0x0028);
		break;
	case RST30:
		newReference |= SetReference(flags, IS_LABEL, 0x0030);
		break;
	case RST38:
		newReference |= SetReference(flags, IS_LABEL, 0x0038);
		break;
	}

	return newReference;

}

#define kCheckAbort 256

/* Disassemble() disassembles static code. */
void
	Disassemble(char *outputFile,
	            Word startAddress,
	            Byte (*RdByte)(Word address))
{
	FILE *output = 0;
	Byte *flags = 0;
	int passCount;
	int referenceCount;
	int newReference;
	Word thisAddress;
	Word nextAddress;
	char buffer[128];
	Byte memory[4];
	int checkAbort;
	int discontinuous;

	/* Open the output file. */
	if ((output = fopen(outputFile, "w")) == 0) {
		printf("?FILE\n");
		goto error;
	}

	/* Allocate 64K of flags, one for each address. */
	if ((flags = malloc(0x10000)) == 0) {
		printf("?MEMORY\n");
		goto error;
	}

	/* Clear the flags for all addresses. */
	memset(flags, 0, 0x10000);

	/* The starting address is a LABEL. */
	flags[startAddress] = IS_LABEL;

	/* Follow all instruction references. */
	checkAbort = 0;
	passCount = 0;
	do {
		printf("\rSEARCHING... %d %04X", ++passCount, startAddress);
		/* Make a pass through memory, note any new references. */
		newReference = 0;
		nextAddress = 0x0000;
		do {
			/* Abort if any key is pressed. */
			if (checkAbort++ > kCheckAbort) {
				if (ConsoleInput(0) != kConsoleNotReady) {
					printf("\nABORT!\n");
					goto error;
				}
				checkAbort = 0;
			}
			thisAddress = nextAddress++;
			/* Follow CODE and LABEL references. */
			if (flags[thisAddress] & (IS_LABEL | IS_CODE)) {
				memory[0] = RdByte(thisAddress + 0);
				memory[1] = RdByte(thisAddress + 1);
				memory[2] = RdByte(thisAddress + 2);
				memory[3] = RdByte(thisAddress + 3);
				newReference |=
					DissasembleReference(flags,
					                     thisAddress,
					                     memory,
					                     &nextAddress);
				if (newReference)
					printf("\b\b\b\b%04X", thisAddress);
			}
			/* Prevent loops, end pass if previous address is next. */
		} while (nextAddress > thisAddress);
		/* All done if no new references were found. */
	} while (newReference != 0);
	printf("\b\b\b\bDONE\n");

	/* Disassemble all referenced code in memory. */
	printf("DISASSEMBLING... 0000");
	nextAddress = 0x0000;
	checkAbort = 0;
	referenceCount = 0;
	discontinuous = 1;
	do {
		/* Abort if any key is pressed. */
		if (checkAbort++ > kCheckAbort) {
			if (ConsoleInput(0) != kConsoleNotReady) {
				printf("\nABORT!\n");
				goto error;
			}
			checkAbort = 0;
		}
		thisAddress = nextAddress++;
		if (flags[thisAddress] == 0)
			discontinuous = 1;
		else {
			printf("\b\b\b\b%04X", thisAddress);
			referenceCount++;
			memory[0] = RdByte(thisAddress + 0);
			memory[1] = RdByte(thisAddress + 1);
			memory[2] = RdByte(thisAddress + 2);
			memory[3] = RdByte(thisAddress + 3);
			/* Is this address referenced as a LABEL? */
			if (flags[thisAddress] & IS_LABEL) {
				if (discontinuous) {
					if (referenceCount != 1)
						fprintf(output, "\n");
					fprintf(output, "     ORG %04X\n", thisAddress);
				}
				DisassembleInstruction(buffer,
				                       thisAddress,
				                       memory,
				                       &nextAddress,
				                       0,
				                       0);
				fprintf(output, "%04X %s\n", thisAddress, buffer);
			}
			/* Is this address referenced as CODE? */
			else if (flags[thisAddress] & IS_CODE) {
				/* Assume not discontinuous. */
				DisassembleInstruction(buffer,
				                       thisAddress,
				                       memory,
				                       &nextAddress,
				                       0,
				                       0);
				fprintf(output, "     %s\n", buffer);
			}
			/* Is this address referenced as DATA? */
			else if (flags[thisAddress] & IS_DATA) {
				if (discontinuous && (referenceCount != 1))
					fprintf(output, "\n");
				fprintf(output, "%04X %02X\n", thisAddress , memory[0]);
			}
			discontinuous = 0;
		}
	} while (nextAddress > thisAddress);
	printf("\b\b\b\bDONE\n");

	/* All done, close the output file and free the flags array. */
error:
	if (output != 0)
		fclose(output);
	if (memory != 0)
		free(flags);

}
