/* uSim cdev.c
 * Copyright (C) 2000, Tsurishaddai Williamson, tsuri@earthlink.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**********************************************************************/

#include "stdio.h"
#include "file.h"
#include "ring.h"
#include <string.h>
#include <stdlib.h>

#include "memory.h"
#include "system.h"
#include "cpu.h"
#include "cdev.h"
#include "monitor.h"

/**********************************************************************/
#pragma mark *** CHARACTER DEVICE ***

#pragma mark gConsoleInputRing
#define kConsoleInputRingSize 256
static Ring gConsoleInputRing;
static char gConsoleInputRingData[kConsoleInputRingSize];

typedef Byte (*CDevStatusFunction)(CDevPtr);
typedef void (*CDevOutputFunction)(CDevPtr, char);
typedef int (*CDevInputFunction)(CDevPtr, char *);
typedef void (*CDevCloseFunction)(CDevPtr);

/* Character Device State Structure. */
#pragma mark struct CDev
struct CDev {
	const char cDev[5];
	const Byte index;
	const Byte mode;
	char name[kMaxFileName];
	int isOpen;
	CDevStatusFunction cDevStatus;
	CDevOutputFunction cDevOutput;
	CDevInputFunction cDevInput;
	CDevCloseFunction cDevClose;
	void *cookie;
};

/* Instantiation of all 16 Chararacter Devices. */
#pragma mark gCDev[]
static CDev gCDev[kMaxCDev] = {
	{ "TTY:", DEVTTY, DEVRW },
	{ "CRT:", DEVCRT, DEVRW },
	{ "UC1:", DEVUC1, DEVRW },
	{ "UC2:", DEVUC2, DEVRW },
	{ "PTR:", DEVPTR, DEVRD },
	{ "UR1:", DEVUR1, DEVRD },
	{ "UR2:", DEVUR2, DEVRD },
	{ "UR3:", DEVUR3, DEVRD },
	{ "PTP:", DEVPTP, DEVWR },
	{ "UP1:", DEVUP1, DEVWR },
	{ "UP2:", DEVUP2, DEVWR },
	{ "UP3:", DEVUP3, DEVWR },
	{ "LPT:", DEVLPT, DEVWR },
	{ "UL1:", DEVUL1, DEVWR },
	{ "UL2:", DEVUL2, DEVWR },
	{ "UL3:", DEVUL3, DEVWR },
};

#pragma mark gIOBYTE[]
struct {
	const char logical[5];
	struct {
		const char physical[5];
	} allowed[4];
	char physical[5];
} gIOBYTE[] = {
	{ "LST:", { "TTY:", "CRT:", "LPT:", "UL1:" }, "" },
	{ "PUN:", { "TTY:", "PTP:", "UP1:", "UP2:" }, "" },
	{ "RDR:", { "TTY:", "PTR:", "UR1:", "UR2:" }, "" },
	{ "CON:", { "TTY:", "CRT:", "BAT:", "UC1:" }, "" }
};

/* IOBYTE() attaches a physical device to a logical device. */
static char *IOBYTE(const char *logical, const char *physical)
{
	unsigned i;
	unsigned j;
	Byte iobyte;
	unsigned shift;
	unsigned mask;

	/* Search for the logical device. */
	for (i = 0; i < 4; i++) {
		if (!strcmp(gIOBYTE[i].logical, logical))
			break;
	}
	if (i >= 4)
		goto error;

	/* Compute the shift and mask for this logical device. */
	/* These will be used to access the bit field in the IOBYTE. */
	shift = (6 - (2 * i));
	mask = 3 << shift;

	/* Attach if a physical device is specified. */
	if (physical != 0) {
		for (j = 0; j < 4; j++) {
			if (!strcmp(gIOBYTE[i].allowed[j].physical, physical))
				break;
		}
		if (j >= 4)
			goto error;
		/* Write the IOBYTE. */
		iobyte = RdByte(3);
		iobyte &= ~mask;
		iobyte |= j << shift;
		WrByte(3, iobyte);
	}

	/* Read the IOBYTE. */
	iobyte = RdByte(3);
	j = (iobyte & mask) >> shift;
	strcpy(gIOBYTE[i].physical, gIOBYTE[i].allowed[j].physical);

	/* All done, no error, return the name of the physical device. */
	return gIOBYTE[i].physical;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* SolicitCDevName() solicits a character device name. */
static int SolicitCDevName(CDevPtr cDevPtr, const char *name)
{
	char defaultName[8];

	/* Prepare the default name.  Example: "LPT.TXT". */
	strcpy(defaultName, cDevPtr->cDev);
	strcpy(strchr(defaultName, ':'), ".TXT");

	/* Get the specified name. */
	strcpy(cDevPtr->name, name);

	/* Solicit a name if none was specified. */
	if (strlen(cDevPtr->name) == 0) {

		if (!gMonitorActive)
			printf(kConsoleCleanLine kConsoleColorSystem);

		printf(CPU "> ATTACH %s [%s]>", cDevPtr->cDev, defaultName);

		if (gets(cDevPtr->name) == 0)
			strcpy(cDevPtr->name, ".");

		if (!gMonitorActive)
			printf(kConsoleColorReset);

	}

	/* Use the default name if none is specified. */
	if (strlen(cDevPtr->name) == 0)
		strcpy(cDevPtr->name, defaultName);

	/* Error if name is ".". */
	if (!strcmp(cDevPtr->name, "."))
		goto error;

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/**********************************************************************/
#pragma mark *** CONSOLE STREAM ***

/* There is no Console Stream State Structure. */

/* ConsoleStreamStatus() returns the status of the console stream. */
static Byte ConsoleStreamStatus(CDevPtr cDevPtr)
{
#pragma unused(cDevPtr)
	Byte status;

	status = DEVWR;

	SystemInterrupt();

	if (RingCount(&gConsoleInputRing) > 0)
		status |= DEVRD;

	return status;

}

/* ConsoleStreamOutput() outputs a character to the console stream. */
static void ConsoleStreamOutput(CDevPtr cDevPtr, char output)
{
#pragma unused(cDevPtr)

	/* Output the character to the console. */
	ConsoleOutput(output);

}

/* ConsoleStreamInput() inputs a character from the console stream. */
static int ConsoleStreamInput(CDevPtr cDevPtr, char *input)
{
#pragma unused(cDevPtr)

	/* Poll for console input... */
	SystemInterrupt();

	/* Remove the input byte from the ring. */
	if (!RingRemove(&gConsoleInputRing, input))
		goto error;

	/* Map DELETE to BACKSPACE. */
	if (*input == DELETEKEY)
		*input = BACKSPACEKEY;

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* ConsoleStreamClose() terminates access to the console stream. */
static void ConsoleStreamClose(CDevPtr cDevPtr)
{

	cDevPtr->cookie = 0;

}

/* ConsoleStreamOpen() prepares access to the console stream. */
static int ConsoleStreamOpen(CDevPtr cDevPtr)
{

	/* Console access must be read/write. */
	if (cDevPtr->mode != DEVRW)
		goto error;

	/* There is no Console Stream State Structure. */
	cDevPtr->cookie = 0;

	/* Reset the Console Input Ring. */
	RingReset(&gConsoleInputRing,
	          gConsoleInputRingData,
	          kConsoleInputRingSize,
	          sizeof(*gConsoleInputRingData));

	/* Set the status, output input and close functions. */
	cDevPtr->cDevStatus = ConsoleStreamStatus;
	cDevPtr->cDevOutput = ConsoleStreamOutput;
	cDevPtr->cDevInput = ConsoleStreamInput;
	cDevPtr->cDevClose = ConsoleStreamClose;

	/* The character device is open. */
	cDevPtr->isOpen = 1;

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	ConsoleStreamClose(cDevPtr);
	return 0;

}

/**********************************************************************/
#pragma mark *** FILE STREAM ***

/* File Stream State Structure. */
#pragma mark struct FileStream
typedef struct FileStream FileStream;
typedef FileStream *FileStreamPtr;
struct FileStream {
	FILE *file;
	int isCRLF;
};

/* FileStreamStatus() returns the status of a file stream. */
static Byte FileStreamStatus(CDevPtr cDevPtr)
{

	return cDevPtr->mode;

}

/* FileStreamOutput() outputs a character to a file stream. */
static void FileStreamOutput(CDevPtr cDevPtr, char output)
{
	FileStreamPtr fileStreamPtr = cDevPtr->cookie;

	/* Close the file stream if Control-Z. */
	if (output == ('Z' - '@'))
		CDevClose(cDevPtr);

	/* Output the character, remove any Carriage Returns. */
	else if (output != '\r') {

		fputc(output, fileStreamPtr->file);

		fflush(fileStreamPtr->file);

	}

}

/* FileStreamInput() inputs a character from a file. */
static int FileStreamInput(CDevPtr cDevPtr, char *input)
{
	FileStreamPtr fileStreamPtr = cDevPtr->cookie;
	int c;

	/*
	 * The CRLF state machine does the following:
	 *    x CR x     =>  x CR x
	 *    x LF x     =>  x CR LF x
	 *    x CR LF x  =>  x CR LF x
	 */

	/* If CRLF state #2... */
	if (fileStreamPtr->isCRLF == 2) {
		/* then set CRLF state #0... */
		fileStreamPtr->isCRLF = 0;
		/* and return LF. */
		*input = '\n';
	}

	/* Error if end of file... */
	else if ((c = fgetc(fileStreamPtr->file)) == EOF)
		goto error;

	/* Error if Control-Z. */
	else if (c == ('Z' - '@'))
		goto error;

	/* If CR... */
	else if (c == '\r') {
		/* then set CRLF state #1... */
		fileStreamPtr->isCRLF = 1;
		/* and return CR. */
		*input = '\r';
	}

	/* If not CR and not LF... */
	else if (c != '\n') {
		/* then set CRLF state #0... */
		fileStreamPtr->isCRLF = 0;
		/* and return the byte. */
		*input = c;
	}

	/* If LF and CRLF state #1... */
	else if (fileStreamPtr->isCRLF == 1) {
		/* then set CRLF state #0... */
		fileStreamPtr->isCRLF = 0;
		/* and return LF. */
		*input = '\n';
	}

	/* If LF and not CRLF state #1... */
	else {
		/* then set CRLF state #2... */
		fileStreamPtr->isCRLF = 2;
		/* and return CR. */
		*input = '\r';
	}

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	CDevClose(cDevPtr);
	return 0;

}

/* FileStreamClose() terminates access to a file stream. */
static void FileStreamClose(CDevPtr cDevPtr)
{
	FileStreamPtr fileStreamPtr = cDevPtr->cookie;

	/* Deallocate the File Stream State Structure. */
	if (fileStreamPtr != 0) {
		if (fileStreamPtr->file != 0)
			fclose(fileStreamPtr->file);
		free(fileStreamPtr);
	}

	cDevPtr->cookie = 0;

}

/* FileStreamOpen() prepares access to a file stream. */
static int FileStreamOpen(CDevPtr cDevPtr, char mode)
{
	FileStreamPtr fileStreamPtr;

	/* Allocate the File Disk State Structure. */
	cDevPtr->cookie = fileStreamPtr =
		malloc(sizeof(FileStream));
	if (fileStreamPtr == 0) {
		SystemMessage("?MALLOC [%s => %s]\n",
		              cDevPtr->cDev,
		              cDevPtr->name);
		goto error;
	}

	/* Try to open the file read/write. */
	if (cDevPtr->mode == DEVRW)
		fileStreamPtr->file = FOpenPath(cDevPtr->name, "r+");

	/* Try to open the file read-only. */
	else if (cDevPtr->mode == DEVRD)
		fileStreamPtr->file = FOpenPath(cDevPtr->name, "r");

	/* Try to open the file write-only, force append. */
	else if (mode == '+')
		fileStreamPtr->file = FOpenPath(cDevPtr->name, "a");

	/* Try to open the file write-only, force replace. */
	else if (mode == '-')
		fileStreamPtr->file = FOpenPath(cDevPtr->name, "w");

	/* Try to open the file write-only, confirm replace. */
	else {
		fileStreamPtr->file = FOpenPath(cDevPtr->name, "r");
		if (fileStreamPtr->file != 0) {
			SystemMessage("?EXISTS [%s => %s]\n",
			              cDevPtr->cDev,
			              cDevPtr->name);
			fclose(fileStreamPtr->file);
			goto error;
		}
		fileStreamPtr->file = FOpenPath(cDevPtr->name, "w");
	}

	/* Error if the fopen() failed. */
	if (fileStreamPtr->file == 0) {
		SystemMessage("?OPEN [%s => %s]\n",
		              cDevPtr->cDev,
		              cDevPtr->name);
		goto error;
	}

	/* Reset the CRLF state. */
	fileStreamPtr->isCRLF = 0;

	/* Set the status, output input and close functions. */
	cDevPtr->cDevStatus = FileStreamStatus;
	cDevPtr->cDevOutput = FileStreamOutput;
	cDevPtr->cDevInput = FileStreamInput;
	cDevPtr->cDevClose = FileStreamClose;

	/* The character device is open. */
	cDevPtr->isOpen = 1;

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	FileStreamClose(cDevPtr);
	return 0;

}

/**********************************************************************/
#pragma mark *** CHARACTER DEVICE ***

/* CDevIndexToPtr() returns a pointer to a character device. */
CDevPtr CDevIndexToPtr(unsigned n)
{

	/* Return zero if the index is out of bounds. */
	return (n < kMaxCDev) ? &gCDev[n] : 0;

}

/* CDevPoll() polls for character device interrupt activity. */
void CDevPoll(void)
{
	char consoleInput;

	/* Poll for console input. */
	switch (consoleInput = ConsoleInput(0)) {

	/* Do nothing if kConsoleNotReady. */
	case kConsoleNotReady:
		break;

	/* Set kSystemHalt flag if kConsoleQuit. */
	case kConsoleQuit:
		SetSystemFlags(kSystemHalt, 0);
		break;

	/* Set kSystemMonitor if kConsoleMonitor. */
	case kConsoleMonitor:
		SetSystemFlags(kSystemMonitor, 0);
		break;

	/* Otherwise, insert the character into the gConsoleInputRing. */
	default:
		(void)RingInsert(&gConsoleInputRing, &consoleInput);
		break;

	}

}

/* CDevStatus() returns the status of a character device. */
Byte CDevStatus(CDevPtr cDevPtr, char *name)
{

	/* Error if the character device is not open. */
	if (!cDevPtr->isOpen)
		goto error;

	/* Return the physical device name if requested. */
	if (name != 0)
		strcpy(name, cDevPtr->name);

	/* All done, return the character device status. */
	return cDevPtr->cDevStatus(cDevPtr);

	/* Return DEVERR if there was an error. */
error:
	return DEVERR;

}

/* CDevOutput() output a byte to a character device. */
void CDevOutput(CDevPtr cDevPtr, Byte output)
{

	/* Output a character. */
	cDevPtr->cDevOutput(cDevPtr, output);

}

/* CDevInput() inputs a byte from a character device. */
Byte CDevInput(CDevPtr cDevPtr)
{
	char input;

	/* If not attached, return ^Z. */
	if (!cDevPtr->isOpen)
		goto error;

	/* Input a character. */
	if (!cDevPtr->cDevInput(cDevPtr, &input))
		goto error;

	/* All done, return the input byte. */
	return (Byte)input;

	/* Return Control-Z if there was an error. */
error:
	return 'Z' - '@';

}

/* CDevClose() terminates access to a character device. */
void CDevClose(CDevPtr cDevPtr)
{

	/* Close the character device if it is open. */
	if (cDevPtr->isOpen != 0)
		cDevPtr->cDevClose(cDevPtr);

	cDevPtr->isOpen = 0;

}

/* CDevOpen() prepares access to an ASCII character device. */
int CDevOpen(CDevPtr cDevPtr, const char *name)
{
	char mode;
	unsigned i;

	/* Error if the character device is already open. */
	if (cDevPtr->isOpen)
		goto error;

	/* Solicit a Character Device Name. */
	if (!SolicitCDevName(cDevPtr, name))
		goto error;

	/* Use prefix '+' to force append. */
	/* Use prefix '-' to force replace. */
	switch (mode = *(cDevPtr->name)) {
	case '+':
	case '-':
		strcpy(cDevPtr->name, &(cDevPtr->name[1]));
		break;
	default:
		mode = 0;
	}

	/* Error if another CDev is already using this name. */
	for (i = 0; i < kMaxCDev; i++) {
		if (i == cDevPtr->index)
			continue;
		if (!strcmp(cDevPtr->name, gCDev[i].name))
			goto error;
	}

	/* Select an open function. */
	if (!strcmp(cDevPtr->name, "CONSOLE")) {
		if (!ConsoleStreamOpen(cDevPtr))
			goto error;
	}
	else {
		if (!FileStreamOpen(cDevPtr, mode))
			goto error;
	}

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* CDevAttach() attaches a file to a character device. */
char *CDevAttach(const char *cDev, const char *name)
{
	unsigned cDevNumber;
	char *result;

	/* Try to attach to a physical cDev. */
	if ((result = IOBYTE(cDev, name)) != 0)
		return result;

	/* Search for the physical cDev. */
	for (cDevNumber = 0; cDevNumber < kMaxCDev; cDevNumber++)
		if (!strcmp(cDev, gCDev[cDevNumber].cDev))
			break;
	if (cDevNumber >= kMaxCDev)
		goto error;

	/* Attach if a file name is specified. */
	if (name != 0) {

		if (gCDev[cDevNumber].isOpen)
			goto error;

		CDevOpen(&gCDev[cDevNumber], name);

		if (!gCDev[cDevNumber].isOpen)
			goto error;

	}

	/* All done, no error, return the name of the attached file. */
	return gCDev[cDevNumber].name;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* CDevDetach() detaches a character device. */
void CDevDetach(const char *cDev)
{
	unsigned cDevNumber;

	/* Search for the cDev to close, all if cDev is 0. */
	for (cDevNumber = 0; cDevNumber < kMaxCDev; cDevNumber++)
		if ((cDev == 0) ||
		    !strcmp(cDev, gCDev[cDevNumber].cDev))
			if (gCDev[cDevNumber].isOpen)
				CDevClose(&gCDev[cDevNumber]);

}

/* ShowCDevAttach() displays attach information for a cDev. */
void ShowCDevAttach(char *cDev)
{

	if (cDev == 0) {
		ShowCDevAttach("CON:");
		ShowCDevAttach("LST:");
		ShowCDevAttach("RDR:");
		ShowCDevAttach("PUN:");
		ShowCDevAttach("TTY:");
		ShowCDevAttach("CRT:");
		ShowCDevAttach("UC1:");
		ShowCDevAttach("UC2:");
		ShowCDevAttach("PTR:");
		ShowCDevAttach("UR1:");
		ShowCDevAttach("UR2:");
		ShowCDevAttach("UR3:");
		ShowCDevAttach("PTP:");
		ShowCDevAttach("UP1:");
		ShowCDevAttach("UP2:");
		ShowCDevAttach("UP3:");
		ShowCDevAttach("LPT:");
		ShowCDevAttach("UL1:");
		ShowCDevAttach("UL2:");
		ShowCDevAttach("UL3:");
	}
	else {
		char *name = CDevAttach(cDev, 0);
		if (name == 0)
			printf("?NODEV [%s]\n", cDev);
		else
			printf("%s => %s\n", cDev, name);
	}

}
